#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmCollectionFacilityMaintain;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_LocalSupplier : BaseTest 
	{
		private Guid _lsGuid;
		private BOL.LocalSupplier _tstLocalSupplier;
		private BOL.LocalSupplier _tstNationalLS;

		private DataTable _dtNational;		
		private DataTable _dtSupplier;
		private DataTable _dtActiveProducts;
		private DataTable _dtActiveUnits;

		[SetUp]
		protected void SetUp() 
		{		
			this.BaseSetUp();

			if (this.RefreshData)
			{		
				DataCreator.CreateBloodUnit(false);		//This should create a LocalSupplier and Shipper, so these other checks aren't needed
				this._dtSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM V_Suppliers WHERE FacilityGuid IS NOT NULL AND AddressGuid IS NOT NULL AND RecordStatusCode = 'L' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

				this._dtNational = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM V_Suppliers WHERE RecordStatusCode = 'N' AND IccbbaRegistrationNumber IS NOT NULL AND FacilityGuid NOT IN (SELECT FacilityGuid FROM LocalSupplier WHERE FacilityGuid IS NOT NULL)", true);

				this._lsGuid = (Guid)this._dtSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];

				//spIsLocalSupplierOnActiveBloodProducts
				this._dtActiveProducts = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM ProductShipper PS INNER JOIN LocalSupplier LS ON PS.LocalSupplierGuid = LS.LocalSupplierGuid WHERE PS.RecordStatusCode != 'X' AND PS.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'",  true);

				//spIsLocalSupplierOnActiveBloodUnits
				this._dtActiveUnits = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE FdaRegistrationNumber IN (SELECT DISTINCT UnitFDARegistrationNumber FROM BloodUnit BU WHERE BU.RecordStatusCode <> 'X' AND BU.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "')", true);

				this.RefreshData = false;
			}

			this._tstNationalLS = new BOL.LocalSupplier(false, this._dtNational.Rows[0][TABLES.Facility.IccbbaRegistrationNumber].ToString());
			this._tstLocalSupplier = new BOL.LocalSupplier(this._lsGuid, false);
			this._tstLocalSupplier.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.NewLocalSupplier);
		}


		[Test]
		public void Ctor_Guid_Bool_Pass()
		{			
			//Existing supplier test
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(this._lsGuid, false);

			Assert.AreEqual(false, tmpLS.IsNew, "Existing-IsNew");
			Assert.AreEqual(this._lsGuid, tmpLS.LocalSupplierGuid, "Existing-LocalSupplierGuid");

			//Test a new supplier
			Guid unitGuid = Guid.NewGuid();
			tmpLS = new BOL.LocalSupplier(unitGuid, false);

			Assert.AreEqual(true, tmpLS.IsNew, "New-IsNew");
			Assert.AreEqual(unitGuid, tmpLS.LocalSupplierGuid, "New-LocalSupplierGuid");
	
			//Code within the LocalSupplier constructor will never be hit
			//Records from the V_Suppliers with "N" RecordStatusCode are Facility-ONLY, no LocalSupplierGuid will exist
			//Unreachable code should be removed
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Bool_Fail()
		{
		}

		[Test]
		public void Ctor_String_Bool_Pass()
		{			
			//Existing supplier test
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(this._tstLocalSupplier.FDARegistrationNumber, false);

			Assert.AreEqual(false, tmpLS.IsNew, "Existing-IsNew");
			Assert.AreEqual(this._lsGuid, tmpLS.LocalSupplierGuid, "Existing-LocalSupplierGuid");
			Assert.AreEqual(this._tstLocalSupplier.FDARegistrationNumber, tmpLS.FDARegistrationNumber, "Existing-FDARegistrationNumber");

			//Test a new supplier
			string tstValue = "0000000000";
			tmpLS = new BOL.LocalSupplier(tstValue, false);
			Assert.AreEqual(true, tmpLS.IsNew, "New-IsNew");
			Assert.AreEqual(tstValue, tmpLS.FDARegistrationNumber, "New-FDARegistrationNumber");

			//Test a national record
			string tstNational = this._dtNational.Rows[0][TABLES.Facility.FdaRegistrationNumber].ToString().Trim();
			tmpLS = new BOL.LocalSupplier(tstNational, false);
			Assert.AreEqual(true, tmpLS.IsNew, "National-IsNew");
			Assert.AreEqual(tstNational, tmpLS.FDARegistrationNumber, "National-FDARegistrationNumber");

		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_String_Bool_Fail()
		{
		}

		[Test]
		public void Ctor_Bool_String_Pass()
		{
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(false, this._tstLocalSupplier.IccbbaRegistrationNumber);
			//
			Assert.IsNotNull(tmpLS, "Old");
			Assert.AreEqual(this._tstLocalSupplier.IccbbaRegistrationNumber, tmpLS.IccbbaRegistrationNumber, "Old-Iccbba#");

			//Test a new IccbbaRegistrationNumber  -- Q seems unused at this time
			string tmpValue = "Q" + new Random().Next(1111,9999).ToString();
			tmpLS = new BOL.LocalSupplier(false, tmpValue);
			Assert.AreEqual(true, tmpLS.IsNew, "New-IsNew");
			Assert.AreEqual(tmpValue, tmpLS.IccbbaRegistrationNumber, "New-Iccbba#");
			
			//Test a National IccbbaRegistrationNumber
			tmpValue = this._dtNational.Rows[0][TABLES.Facility.IccbbaRegistrationNumber].ToString();
			tmpLS = new BOL.LocalSupplier(false, tmpValue);
			Assert.AreEqual(true, tmpLS.IsNew, "National-IsNew");
			Assert.AreEqual(tmpValue, tmpLS.IccbbaRegistrationNumber, "National-Iccbba#");
			
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Bool_String_Fail()
		{
		}

		[Test]
		public void GetLocalSupplierLookupList_Pass()
		{
			string facilityName = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FacilityName].ToString();
			string fdaRegNum = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString();

			DataTable dtTest = BOL.LocalSupplier.GetLocalSupplierLookupList(facilityName, fdaRegNum);
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetLocalSupplierLookupList_Fail()
		{
			DataTable dtTest = BOL.LocalSupplier.GetLocalSupplierLookupList("Count Chocula", "     enberry");
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");

		}

		[Test]
		public void GetFacilityNationalLookupList_Pass()
		{
			DataTable dtTest = BOL.LocalSupplier.GetFacilityNationalLookupList();
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetFacilityNationalLookupList_Fail() 
		{
		}

		[Test]
		public void GetLocalSupplierDetailByFdaRegNum_Pass()
		{
			string facilityName = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FacilityName].ToString();

			DataTable dtTest = this._tstLocalSupplier.GetLocalSupplierDetailByFdaRegNum(this._tstLocalSupplier.FDARegistrationNumber, false);
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetLocalSupplierDetailByFdaRegNum_Fail()
		{
			DataTable dtTest = this._tstLocalSupplier.GetLocalSupplierDetailByFdaRegNum(string.Empty , false);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetLocalSuppliersForDivision_Pass()
		{
			DataTable dtTest = BOL.LocalSupplier.GetLocalSuppliersForDivision();
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetLocalSuppliersForDivision_Fail()
		{
		}

		[Test]
		public void IsValidFdaRegistrationNumber_Pass()
		{
			string fdaRegNum = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString();

			bool retValue = BOL.LocalSupplier.IsValidFdaRegistrationNumber(fdaRegNum);
			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsValidFdaRegistrationNumber_Fail()
		{
			bool retValue = BOL.LocalSupplier.IsValidFdaRegistrationNumber(string.Empty);
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void InsertLocalSupplier_Pass()
		{
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(this._tstLocalSupplier.LocalSupplierGuid, false);

			//10 Times should be enough to find us a valid #
			for (int ndx=0; ndx < 10; ndx++)
			{
				tmpLS.FDARegistrationNumber = new Random().Next(1111111,9999999).ToString();
				if (!BOL.LocalSupplier.IsValidFdaRegistrationNumber(tmpLS.FDARegistrationNumber))
					break;
			}
			
			tmpLS.LocalSupplierGuid = Guid.NewGuid();

			bool retValue = tmpLS.InsertLocalSupplier(Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.UniqueDatabaseConstraintViolatedException))]
		public void InsertLocalSupplier_Fail()
		{
            this._tstLocalSupplier.InsertLocalSupplier(Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void UpdateLocalSupplier_Pass()
		{
			this.RefreshData = true;
			this._tstLocalSupplier.FacilityName = "Indys Stabbin Hut";
			bool retValue = this._tstLocalSupplier.UpdateLocalSupplier(Common.UpdateFunction.UnitTests);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UpdateLocalSupplier_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataTable dtTemp = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(false);
			dtTemp.Columns.Add(TABLES.Address.AddressCity, typeof(string));
			dtTemp.Columns.Add(TABLES.Address.AddressLine1, typeof(string));
			dtTemp.Columns.Add(TABLES.Address.AddressLine2, typeof(string));
			dtTemp.Columns.Add(TABLES.Address.AddressLine3, typeof(string));
			dtTemp.Columns.Add(TABLES.Address.AddressZipCode, typeof(string));
			dtTemp.Columns.Add(TABLES.Address.StateCode, typeof(string));
			dtTemp.Columns.Add(ARTIFICIAL.AddressRowVersion, typeof(byte[]));

			DataRow dr = this._tstLocalSupplier.LoadDataRowFromThis(dtTemp.NewRow());
			Assert.AreEqual(this._tstLocalSupplier.LocalSupplierGuid, (Guid)dr[TABLES.LocalSupplier.LocalSupplierGuid], "LocalSupplierGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		public void IsNewFDARegistrationNumber_Pass()
		{
			bool retValue = BOL.LocalSupplier.IsNewFDARegistrationNumber(this._tstLocalSupplier.FDARegistrationNumber);
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void IsNewFDARegistrationNumber_Fail()
		{
			bool retValue = BOL.LocalSupplier.IsNewFDARegistrationNumber("!DUFFBEER!");
			Assert.AreEqual(true, retValue, "ReturnValue");
		}


		[Test]
		public void LocalSupplierGuid_Pass()
		{
			Guid tstGuid = Guid.NewGuid();

			this._tstLocalSupplier.LocalSupplierGuid = tstGuid;
			Assert.AreEqual(tstGuid, this._tstLocalSupplier.LocalSupplierGuid, "LocalSupplierGuid");

			//National records have NULL for the LocalSupplierGuid
			Guid lsGuid = this._tstNationalLS.LocalSupplierGuid;
			Assert.AreEqual(lsGuid, this._tstNationalLS.LocalSupplierGuid, "Null Column");
		}

		[Test]
		public void LocalSupplierGuid_Fail()
		{
			this._tstLocalSupplier.LocalSupplierGuid = Guid.Empty;
			Assert.AreEqual(true, this._tstLocalSupplier.IsRuleBroken(BROKEN.LocalSupplierGuid), "BrokenRule");	
		}

		[Test]
		public void FacilityGuid_Pass()
		{
			Guid tstGuid = Guid.NewGuid();
			this._tstLocalSupplier.FacilityGuid = tstGuid;

			Assert.AreEqual(tstGuid, this._tstLocalSupplier.FacilityGuid, "Valid Value");

			//Test null column
			DataTable dtLS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE FacilityGuid IS NOT NULL AND RecordStatusCode <> 'X' AND DivisionCode = '" + this.TestDivisionCode + "'");
			Guid lsGuid = (Guid)dtLS.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];
			Guid fGuid = (Guid)dtLS.Rows[0][TABLES.Facility.FacilityGuid];

			//Set the FacilityGuid to NULL
			UnitTestUtil.RunSQL("UPDATE LocalSupplier SET FacilityGuid = NULL WHERE LocalSupplierGuid = '" + lsGuid.ToString() + "'");

			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(lsGuid, false); 
			Assert.AreEqual(Guid.Empty, tmpLS.FacilityGuid, "Null Column");

			//Set things back the way they were
			UnitTestUtil.RunSQL("UPDATE LocalSupplier SET FacilityGuid = '" + fGuid.ToString() + "' WHERE LocalSupplierGuid = '" + lsGuid.ToString() + "'");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void FacilityGuid_Fail()
		{
		}

		[Test]
		public void FacilityName_Pass()
		{
			string tstName = "Test Facility Name";
			this._tstLocalSupplier.FacilityName = tstName;

			Assert.AreEqual(tstName, this._tstLocalSupplier.FacilityName, "FacilityName");

			int brCount = this._tstLocalSupplier.BrokenRulesCount;
			this._tstLocalSupplier.FacilityName = string.Empty;
			Assert.AreEqual(brCount+1, this._tstLocalSupplier.BrokenRulesCount, "BrokenRuleCount");

			//Cannot test code where FacilityName IS NULL - unreachable code, because it is a required field
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void FacilityName_Fail()
		{
			this._tstLocalSupplier.FacilityName = "".PadRight(121,'X');
		}

		[Test]
		public void FDARegistrationNumber_Pass()
		{
			string tstFDA = "1234567";
			this._tstLocalSupplier.FDARegistrationNumber = tstFDA;

			Assert.AreEqual(tstFDA, this._tstLocalSupplier.FDARegistrationNumber, "Valid FDARegistrationNumber");	

			int brCount = this._tstLocalSupplier.BrokenRulesCount;
			this._tstLocalSupplier.FDARegistrationNumber = string.Empty;
			Assert.AreEqual(brCount+1, this._tstLocalSupplier.BrokenRulesCount, "Empty String");

		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void FDARegistrationNumber_Fail()
		{
			this._tstLocalSupplier.FDARegistrationNumber = "Wiggum";
		}

		[Test]
		public void FacilityPrefix_Pass()
		{
			string tstPrefix = "99";
			this._tstLocalSupplier.FacilityPrefix = tstPrefix;
			Assert.AreEqual(tstPrefix, this._tstLocalSupplier.FacilityPrefix, "FacilityPrefix");

			//Test null column - National Records don't have Prefix's at this time
			tstPrefix = this._tstNationalLS.FacilityPrefix;
			Assert.AreEqual(string.Empty, tstPrefix, "Null Column");

			int brCount = this._tstLocalSupplier.BrokenRulesCount;
			this._tstLocalSupplier.FacilityPrefix = "1";
			Assert.AreEqual(brCount + 1, this._tstLocalSupplier.BrokenRulesCount, "1 Char");

			this._tstLocalSupplier.FacilityPrefix = "123";
			Assert.AreEqual(brCount + 1, this._tstLocalSupplier.BrokenRulesCount, ">2 Chars");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void FacilityPrefix_Fail()
		{
			//FacilityPrefix msut be numeric
			this._tstLocalSupplier.FacilityPrefix = "AB";
			
		}

		[Test]
		public void IccbbaRegistrationNumber_Pass()
		{
			string rememberIccbba = this._tstLocalSupplier.IccbbaRegistrationNumber;
			
			this._tstLocalSupplier.IccbbaRegistrationNumber = "W1234";
			Assert.AreEqual("W1234",this._tstLocalSupplier.IccbbaRegistrationNumber,"IccbbaRegistrationNumber");
			
			//National records don't have IccbbaRegistration #'s at this time
            DataRow drLS = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 FdaRegistrationNumber FROM Facility WHERE FdaRegistrationNumber IS NOT NULL");
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(drLS[TABLES.LocalSupplier.FdaRegistrationNumber].ToString(), false);
			Assert.AreEqual(string.Empty, tmpLS.IccbbaRegistrationNumber, "Null Column" );
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void IccbbaRegistrationNumber_Fail()
		{
			this._tstLocalSupplier.IccbbaRegistrationNumber = "INVALID FORMAT";
		}

		[Test]
		public void LabelUnitIDTrimIndicator_Pass()
		{
			bool tstValue = (!this._tstLocalSupplier.LabelUnitIDTrimIndicator);
			this._tstLocalSupplier.LabelUnitIDTrimIndicator = tstValue;

			Assert.AreEqual(tstValue, this._tstLocalSupplier.LabelUnitIDTrimIndicator, "LabelUnitIDTrimIndicator");

			//National records have NULL for the LabelUnitIdTrimIndicator
			Assert.AreEqual(false, this._tstNationalLS.LabelUnitIDTrimIndicator, "Null Column");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LabelUnitIDTrimIndicator_Fail()
		{
		}

		[Test]
		public void PhoneNumber_Pass()
		{
			string tstPhone = "(123)456-7890";
			this._tstLocalSupplier.PhoneNumber = tstPhone;

			Assert.AreEqual(tstPhone, this._tstLocalSupplier.PhoneNumber, "PhoneNumber");

			//National records have NULL for the PhoneNumber
			Assert.AreEqual(string.Empty, this._tstNationalLS.PhoneNumber, "Null Column");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void PhoneNumber_Fail()
		{
			this._tstLocalSupplier.PhoneNumber = "".PadRight(15,'X');
		}


		[Test]
		public void FaxNumber_Pass()
		{
			string tstFax = "(123)456-7890";
			this._tstLocalSupplier.FaxNumber = tstFax;

			Assert.AreEqual(tstFax, this._tstLocalSupplier.FaxNumber, "FaxNumber");

			//National records don't contain fax numbers
			Assert.AreEqual(string.Empty, this._tstNationalLS.FaxNumber, "National-FaxNumber");

		}

		[Test]
		public void FaxNumber_Fail()
		{
			try
			{
				this._tstLocalSupplier.FaxNumber = "".PadRight(15,'X');
				Assert.Fail("FaxNumber.Length > 14 passed!");
			}
			catch(BOL.BusinessObjectException)
			{
				Assert.IsTrue(true, "FaxNumber.Length > 14");
			}

			try
			{
				this._tstLocalSupplier.FaxNumber = "".PadRight(13,'X');
				Assert.Fail("Invalid FaxNumber passed!");
			}
			catch(BOL.BusinessObjectException)
			{
				Assert.IsTrue(true, "Invalid FaxNumber");
			}
		}

		[Test]
		public void Address_Pass()
		{
			string tstValue = new Random().Next(11111,55555).ToString() + " Evergreen Terrace";

			this._tstLocalSupplier.Address = new BOL.Address();
			this._tstLocalSupplier.Address.AddressLine1 = tstValue;

			Assert.AreEqual(tstValue, this._tstLocalSupplier.Address.AddressLine1, "AddressLine1");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Address_Fail()
		{
		}

		[Test]
		public void HasAddress_Pass()
		{
			//We know its going to have an address, because we selected it that way
			Assert.AreEqual(true, this._tstLocalSupplier.HasAddress, "HasAddress");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasAddress_Fail()
		{
		}

		[Test]
		public void IsCollectionFacility_Pass()
		{
			bool tstValue = (!this._tstLocalSupplier.IsCollectionFacility);
			
			this._tstLocalSupplier.IsCollectionFacility = tstValue;

			Assert.AreEqual(tstValue, this._tstLocalSupplier.IsCollectionFacility, "IsCollectionFacility");

			//National records have NULL for the CollectionIndicator
			Assert.AreEqual(true, this._tstNationalLS.IsCollectionFacility, "Null Column");
		}

		[Test]
		public void IsCollectionFacility_Fail()
		{
			this._tstLocalSupplier.IsTestingFacility = false;
			this._tstLocalSupplier.IsCollectionFacility = false;

			Assert.AreEqual(true, this._tstLocalSupplier.IsRuleBroken(BROKEN.FacilityIndicator), "BrokenRule");
		}

		[Test]
		public void IsTestingFacility_Pass()
		{
			bool tstValue = (!this._tstLocalSupplier.IsTestingFacility);
			
			this._tstLocalSupplier.IsTestingFacility = tstValue;

			Assert.AreEqual(tstValue, this._tstLocalSupplier.IsTestingFacility, "IsTestingFacility");

			//National records have NULL for the CollectionIndicator
			Assert.AreEqual(false, this._tstNationalLS.IsTestingFacility, "Null Column");
		}

		[Test]
		public void IsTestingFacility_Fail()
		{
			this._tstLocalSupplier.IsCollectionFacility = false;
			this._tstLocalSupplier.IsTestingFacility = false;

			Assert.AreEqual(true, this._tstLocalSupplier.IsRuleBroken(BROKEN.FacilityIndicator), "BrokenRule");
		}


		[Test]
		public void IsLocalSupplierOnActiveBloodProducts_Pass()
		{
			Guid unitGuid = (Guid)this._dtActiveProducts.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];

			bool retValue = BOL.LocalSupplier.IsLocalSupplierOnActiveBloodProducts(unitGuid);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodProducts_Fail()
		{
			Guid unitGuid = Guid.NewGuid();

			bool retValue = BOL.LocalSupplier.IsLocalSupplierOnActiveBloodProducts(unitGuid);

			Assert.AreEqual(false, retValue, "ReturnValue");
		}


		[Test]
		public void IsLocalSupplierOnActiveBloodUnits_Pass()
		{
			string fdaRegNum = this._dtActiveUnits.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString();

			bool retValue = BOL.LocalSupplier.IsLocalSupplierOnActiveBloodUnits(fdaRegNum);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodUnits_Fail()
		{
			string fdaRegNum = "Smelly";

			bool retValue = BOL.LocalSupplier.IsLocalSupplierOnActiveBloodUnits(fdaRegNum);

			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierActiveByIccbbaRegistrationNumber_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 IccbbaRegistrationNumber FROM dbo.LocalSupplier WHERE RecordStatusCode = 'L' AND DivisionCode = '"+UnitTestConfig.TestDivisionCode+"'");
			//
			Assert.AreEqual(true,BOL.LocalSupplier.IsLocalSupplierActiveByIccbbaRegistrationNumber(dr[TABLES.LocalSupplier.IccbbaRegistrationNumber].ToString()),"IsLocalSupplier");
		}

		[Test]
		public void IsLocalSupplierActiveByIccbbaRegistrationNumber_Fail()
		{
			Assert.AreEqual(false,BOL.LocalSupplier.IsLocalSupplierActiveByIccbbaRegistrationNumber("INVALID ICCBBA"),"NotLocalSupplier");
		}

		[Test]
		public void RecordStatusCode_Pass()
		{
			this._tstLocalSupplier.RecordStatusCode = Common.RecordStatusCode.Inactive;

			Assert.AreEqual(Common.RecordStatusCode.Inactive, this._tstLocalSupplier.RecordStatusCode, "RecordStatusCode");	

			//Cannot test for NULL RecordStatuscode
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RecordStatusCode_Fail()
		{
		}
	}
}
#endif






